<?php namespace Barryvdh\Snappy;

use Illuminate\View\View;
use PHPUnit\Framework\Assert as PHPUnit;
use Illuminate\Support\Facades\View as ViewFacade;


class PdfFaker extends PdfWrapper
{
	protected $view;
	protected $filename;

	/**
     * Load a View and convert to HTML
     *
     * @param  string $view
     * @param  array $data
     * @param  array $mergeData
     * @return $this
     */
    public function loadView($view, $data = array(), $mergeData = array())
    {
        $this->view = ViewFacade::make($view, $data, $mergeData);
        return parent::loadView($view, $data, $mergeData);
    }


    /**
     * Ensure that the response has a view as its original content.
     *
     * @return $this
     */
    protected function ensureResponseHasView()
    {
        if (! isset($this->view) || ! $this->view instanceof View) {
            return PHPUnit::fail('The response is not a view.');
        }

        return $this;
    }

    public function assertViewIs($value)
    {
    	PHPUnit::assertEquals($value, $this->view->getName());

    	return $this;
    }

    /**
     * Assert that the response view has a given piece of bound data.
     *
     * @param  string|array  $key
     * @param  mixed  $value
     * @return $this
     */
    public function assertViewHas($key, $value = null)
    {
        if (is_array($key)) {
            return $this->assertViewHasAll($key);
        }

        $this->ensureResponseHasView();

        if (is_null($value)) {
            PHPUnit::assertArrayHasKey($key, $this->view->getData());
        } elseif ($value instanceof \Closure) {
            PHPUnit::assertTrue($value($this->view->$key));
        } else {
            PHPUnit::assertEquals($value, $this->view->$key);
        }

        return $this;
    }

    /**
     * Assert that the response view has a given list of bound data.
     *
     * @param  array  $bindings
     * @return $this
     */
    public function assertViewHasAll(array $bindings)
    {
        foreach ($bindings as $key => $value) {
            if (is_int($key)) {
                $this->assertViewHas($value);
            } else {
                $this->assertViewHas($key, $value);
            }
        }

        return $this;
    }

    /**
     * Assert that the response view is missing a piece of bound data.
     *
     * @param  string  $key
     * @return $this
     */
    public function assertViewMissing($key)
    {
        $this->ensureResponseHasView();

        PHPUnit::assertArrayNotHasKey($key, $this->view->getData());

        return $this;
    }

    /**
     * Assert that the given string is contained within the response.
     *
     * @param  string  $value
     * @return $this
     */
    public function assertSee($value)
    {
        PHPUnit::assertContains($value, $this->html);

        return $this;
    }

    /**
     * Assert that the given string is contained within the response text.
     *
     * @param  string  $value
     * @return $this
     */
    public function assertSeeText($value)
    {
        PHPUnit::assertContains($value, strip_tags($this->html));

        return $this;
    }

    /**
     * Assert that the given string is not contained within the response.
     *
     * @param  string  $value
     * @return $this
     */
    public function assertDontSee($value)
    {
        PHPUnit::assertNotContains($value, $this->html);

        return $this;
    }

    /**
     * Assert that the given string is not contained within the response text.
     *
     * @param  string  $value
     * @return $this
     */
    public function assertDontSeeText($value)
    {
        PHPUnit::assertNotContains($value, strip_tags($this->html));

        return $this;
    }
    
    /**
     * Assert that the given string is equal to the saved filename.
     *
     * @param  string  $value
     * @return $this
     */
    public function assertFileNameIs($value)
    {
    	PHPUnit::assertEquals($value, $this->filename);

    	return $this;
    }

    public function output()
    {
        return '%PDF-1.3
%?????????
4 0 obj
<< /Length 5 0 R /Filter /FlateDecode >>
stream
x+T(T0BSKS=#

            C=
              K??T?p?<}?bC??bC0,N??5?34???05j1?7)N?
z/?
endstream
endobj
5 0 obj
73
endobj
2 0 obj
<< /Type /Page /Parent 3 0 R /Resources 6 0 R /Contents 4 0 R /MediaBox [0 0 595.28 841.89]
>>
endobj
6 0 obj
<< /ProcSet [ /PDF ] /ColorSpace << /Cs1 7 0 R >> >>
endobj
8 0 obj
<< /Length 9 0 R /N 3 /Alternate /DeviceRGB /Filter /FlateDecode >>
stream
x??wTS??Ͻ7??" %?z	 ?;HQ?I?P??&vDF)VdT?G?"cE
                                                 ??b?	?P??QDE?݌k	?5?ޚ??Y?????g?}׺P???tX?4?X???\???X??ffG?D???=???HƳ??.?d??,?P&s???"7C$
E?6<~&??S??2????)2?12?	??"?įl???+?ɘ?&?Y??4???Pޚ%ᣌ?\?%?g?|e?TI???(????L0?_??&?l?2E???9?r??9h?x?g??Ib?טi???f??S?b1+??M?xL???
                                           ?0??o?E%Ym?h?????Y??h????~S?=?z?U?&?ϞA??Y?l?/??$Z????U?m@??O? ??ޜ??l^???
                                   \'
                                    ???ls?k.+?7???oʿ?9?????V;???#I3eE妧?KD??
                                                                            ??d?????9i???,?????UQ?	??h??<?X?.d
???6\'~?khu_}?9P?I?o=C#$n?z}?[1
Ⱦ?h???s?2z???
             \?n?LA"S??
                       ?dr%?,?߄l??t?
?3p? ??H?.Hi@?A>?
p1?v?jpԁz?N?6p\W?

 ?G@
???ٰG???Dx????J?>???,?_@?FDB?X$!k?"??E????H?q???a???Y??bVa?bJ0՘c?VL?6f3????bձ?X\'??&?x?*???s?b|!??`[????a?;???p~?\2n5??׌????
ߏƿ\'?	Zk?!? $l$???4Q??Ot"?y?\b)???A?I&N?I?$R$)???TIj"]&=&?!??:dGrY@^O?$? _%??P?n?X????ZO?D}J}/G?3???ɭ???k??{%O?חw?_.?\'_!J????Q?@?S???V?F??=?IE???b?b?b?b??5?Q%?????O?@??%?!BӥyҸ?M?:?e?0G7??ӓ?????	e%e[?(????R?0`?3R????????4?????6?i^??)??*n*|?"?f????LUo?՝?m?O?0j&jaj?j??.??ϧ?w?ϝ_4????갺?z??j???=???U?4?5?n?ɚ??4ǴhZ
                                                                           ?Z?Z?^0????Tf%??9?????-?>?ݫ=?c??Xg?N??]?.[7A?\?SwBOK/X/_?Q?>Q?????G?[??? ?`?A???????a?a??c#????*?Z?;?8c?q??>?[&???I?I??MS???T`?ϴ?
                                            k?h&4?5?Ǣ??YY?F֠9?<?|?y??+
                                                                     =?X???_,?,S-?,Y)YXm?????Ěk]c}ǆj?c?Φ?浭?-?v??};?]???N????"?&?1=?x????tv(??}???????\'{\'??I?ߝY?Σ?

 ??-r?q?r?.d.?_xp??Uە?Z???M׍?v?m???=????+K?G?ǔ????
                                                  ^???W?W????b?j?>:>?>?>?v??}/?a??v?????????O8?	?
?FV>
    2	u?????/?_$\?B?Cv?<	5
                                 ]?s.,4?&?y?Ux~xw-bEDCĻH????G??KwF?G?E?GME{E?EK?X,Y??F?Z? ?=$vr????K????
??.3\????r???Ϯ?_?Yq*??©?L??_?w?ד??????+??]?e???????D??]?cI?II?OA??u?_?䩔???)3?ѩ?i?????B%a??+]3=\'?/?4?0C??i??U?@ёL(sYf????L?H?$?%?Y
                                                  ?j??gGe??Q?????n?????~5f5wug?v????5?k??֮\۹Nw]??????m mH???Fˍen???Q?Q??`h????B?BQ?-?[l?ll??f??jۗ"^??b???O%ܒ??Y}W???????????w?w????X?bY^?Ю?]?????W?Va[q`i?d??2???J?jGէ??????{?????׿?m???>???Pk?Am?a?????꺿g_D?H??G?G??u?;??7?7?6?Ʊ?q?o???C{??P3???8!9?????
                                                          <?y?}??\'?????Z?Z???։??6i{L{??ӝ?-???|??????gKϑ???9?w~?Bƅ??:Wt>???ҝ????ˁ??^?r?۽??U??g?9];}?}????????_?~i??m??p???㭎?}??]?/???}?????.?{?^?=?}????^??z8?h?c??\'
O*????????f?????`ϳ?g???C/????O?ϩ?+F?F?G?Gό???z????ˌ??ㅿ)????ѫ?~w??gb???k???Jި?9???m?d???wi獵?ޫ???????c?Ǒ??O?O????w|	??x&mf??????
endstream
endobj
9 0 obj
2612
endobj
7 0 obj
[ /ICCBased 8 0 R ]
endobj
3 0 obj
<< /Type /Pages /MediaBox [0 0 595.28 841.89] /Count 1 /Kids [ 2 0 R ] >>
endobj
10 0 obj
<< /Type /Catalog /Pages 3 0 R >>
endobj
11 0 obj
(Leeg)
endobj
12 0 obj
(Mac OS X 10.13.3 Quartz PDFContext)
endobj
13 0 obj
(Pages)
endobj
14 0 obj
(D:20180330091154Z00\'00\')
endobj
1 0 obj
<< /Title 11 0 R /Producer 12 0 R /Creator 13 0 R /CreationDate 14 0 R /ModDate
14 0 R >>
endobj
xref
0 15
0000000000 65535 f 
0000003414 00000 n 
0000000187 00000 n 
0000003133 00000 n 
0000000022 00000 n 
0000000169 00000 n 
0000000297 00000 n 
0000003098 00000 n 
0000000365 00000 n 
0000003078 00000 n 
0000003222 00000 n 
0000003272 00000 n 
0000003295 00000 n 
0000003348 00000 n 
0000003372 00000 n 
trailer
<< /Size 15 /Root 10 0 R /Info 1 0 R /ID [ <ea5d2c625a19688ce1ff05174d8a0261>
<ea5d2c625a19688ce1ff05174d8a0261> ] >>
startxref
3519
%%EOF';
    }

    /**
     * Save the PDF to a file
     *
     * @param  $filename
     * @return $this
     */
    public function save($filename, $overwrite = false)
    {
	$this->filename = $filename;
        return $this;
    }
}
